package oracle.spatial.georaster.samples;

import java.awt.BorderLayout;
import java.awt.Dimension;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.HeadlessException;
import java.awt.Insets;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.KeyEvent;

import java.io.FileInputStream;
import java.io.FileOutputStream;

import java.sql.Connection;

import java.util.Properties;

import javax.swing.JButton;
import javax.swing.JDialog;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JMenu;
import javax.swing.JMenuBar;
import javax.swing.JMenuItem;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JPasswordField;
import javax.swing.JTextField;
import javax.swing.KeyStroke;

import oracle.jdbc.pool.OracleDataSource;

import oracle.spatial.georaster.JGeoRaster;

import oracle.sql.NUMBER;


/**
 * These Java sample files are used to demonstrate how to build applications based on GeoRaster
 * Java API library to view/load/export GeoRaster objects stored in the Oracle database. 
 * 
 * The Tools class is an access point for all samples, and it provides GUI interface for user to  
 * invoke three tools below:
 * <pre>
 *        Viewer
 *        Loader
 *        Exporter
 * </pre>
 * <p>
 */
public class Tools extends JFrame implements ActionListener
{
    static final String RASTERIMAGE = "Rasters";
    static final String LOADFROMDB = "Retrieve from DB";
    static final String EXIT = "Exit";
    static final String IO = "I/O";
    static final String IMPORT = "Import";
    static final String EXPORT = "Export";
    static final String ABOUT = "About";
    
    private Connection m_conn = null;  
    private Viewer m_viewer;
    private JGeoRaster m_jgeor=null;
    
    public Tools(String title)
    {
        super(title);
        init();
    }
    
    //initialize GUI interface
    public void init()
    {
        this.setPreferredSize(new Dimension(800,600));
        
        //initialize menu
        //Create the menu bar.
        JMenuBar jmenuBar = new JMenuBar();

        //Build RasterImage menu.
        JMenu menu = new JMenu(Tools.RASTERIMAGE);
        menu.setMnemonic(KeyEvent.VK_A);
        menu.getAccessibleContext().setAccessibleDescription(Tools.RASTERIMAGE);
        jmenuBar.add(menu);

        //a group of JMenuItems of RasterImage menu.
        JMenuItem menuItem = new JMenuItem(Tools.LOADFROMDB, KeyEvent.VK_E);
        menuItem.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_E,ActionEvent.ALT_MASK));
        menuItem.getAccessibleContext().setAccessibleDescription(Tools.LOADFROMDB);
        menuItem.addActionListener(this);
        menu.add(menuItem);

        menu.addSeparator();

        menuItem = new JMenuItem(Tools.EXIT, KeyEvent.VK_X);
        menuItem.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_X,ActionEvent.ALT_MASK));
        menuItem.getAccessibleContext().setAccessibleDescription(Tools.EXIT);
        menuItem.addActionListener(this);
        menu.add(menuItem);  

        //Build IO menu.
        menu = new JMenu(Tools.IO);
        jmenuBar.add(menu);
        menuItem = new JMenuItem(Tools.IMPORT, KeyEvent.VK_I);
        menuItem.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_I,ActionEvent.ALT_MASK));
        menuItem.getAccessibleContext().setAccessibleDescription(Tools.IMPORT);
        menuItem.addActionListener(this);
        menu.add(menuItem);
        menuItem = new JMenuItem(Tools.EXPORT, KeyEvent.VK_P);
        menuItem.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_P,ActionEvent.ALT_MASK));
        menuItem.getAccessibleContext().setAccessibleDescription(Tools.EXPORT);
        menuItem.addActionListener(this);
        menu.add(menuItem);        

        //Build About menu.
        menu = new JMenu(Tools.ABOUT);
        jmenuBar.add(menu);
        menuItem = new JMenuItem(Tools.ABOUT, KeyEvent.VK_B);
        menuItem.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_B,ActionEvent.ALT_MASK));
        menuItem.getAccessibleContext().setAccessibleDescription(Tools.ABOUT);
        menuItem.addActionListener(this);
        menu.add(menuItem);
        
        setJMenuBar(jmenuBar);
        
        getContentPane().setLayout(new GridBagLayout());
        JPanel mainPanel=new JPanel();
        m_viewer = new Viewer();
        getContentPane().add(m_viewer, new GridBagConstraints(
            0, //gridx
            0, //gridy
            1, //gridwidth
            1, //gridlength
            1.0D, //weightx
            1.0D, //weighty
            GridBagConstraints.NORTHWEST, //anchor
            GridBagConstraints.BOTH, //fill
            new Insets(0, 0, 0, 0), //insets
            0, //ipadx
            0 //ipady
            ));
        
    }
    
    public void actionPerformed(ActionEvent e)
    {
        String title="";
        if(e.getSource() instanceof JMenuItem)
        {
            title = ((JMenuItem) e.getSource()).getText();
            if (title.equals(Tools.ABOUT))
            {
              JOptionPane.showMessageDialog(this, "Oracle Spatial GeoRaster Sample Tools.",
                                            "About: Oracle Spatial GeoRaster Sample Tools",
                                            JOptionPane.INFORMATION_MESSAGE);

            }            
            else if (title.equals(Tools.EXIT))
            {
              System.exit(0);
            }     
            else if (title.equals(Tools.IMPORT))
            {
                Loader loader=new Loader(this);
                loader.showDialog();
            }     
            else if (title.equals(Tools.EXPORT))
            {
                if(m_jgeor==null)
                {
                    JOptionPane.showConfirmDialog(this, "No GeoRaster object is displayed.",
                                                  "Error!", JOptionPane.DEFAULT_OPTION);                
                    return;
                }
                Exporter exporter=new Exporter(m_jgeor);
                exporter.showDialog();
            }                 
            else if (title.equals(Tools.LOADFROMDB))
            {
                GeoDBDialog dialog = new GeoDBDialog(this);
                dialog.show();
                if (dialog.m_ret) 
                {
                    String hostName = dialog.m_hostName.getText().trim();
                    String port = dialog.m_port.getText().trim();
                    String dbName = dialog.m_DBName.getText().trim();
                    String user = dialog.m_userId.getText().trim();
                    String password = dialog.m_password.getText().trim();
                    String rdt = dialog.m_rdt.getText().trim();
                    String rid = dialog.m_rid.getText().trim();
                    try
                    {
                        try
                        {
                            // Create a OracleDataSource instance
                            OracleDataSource ods = new OracleDataSource();
                            // Set the driver type
                            ods.setDriverType("thin");
                            // Set the database server name
                            ods.setServerName(hostName);
                            // Set the database name
                            ods.setDatabaseName(dbName);
                            // Set the port number
                            ods.setPortNumber(new Integer( (String) port).intValue());
                            // Set the user name
                            ods.setUser(user);
                            // Set the password
                            ods.setPassword(password);
                            m_conn = ods.getConnection();
                        }
                        catch(Exception ee)
                        {
                            //try service name
                            String url="jdbc:oracle:thin:@(DESCRIPTION=(LOAD_BALANCE=on)(ADDRESS_LIST="+
                                "(ADDRESS=(PROTOCOL=TCP)(HOST="+hostName+") (PORT="+port+")))"+
                                "(CONNECT_DATA=(SERVICE_NAME="+dbName+")))";
                            try
                            {
                                OracleDataSource ods = new OracleDataSource();
                                ods.setDriverType("thin");
                                ods.setURL(url);
                                ods.setUser(user);
                                ods.setPassword(password);
                                m_conn=ods.getConnection();
                                m_conn.setAutoCommit(false);
                            }
                            catch(Exception eee)
                            {
                                eee.printStackTrace();
                                JOptionPane.showConfirmDialog(this, "Can't make a connection to DB",
                                                            "Error!", JOptionPane.DEFAULT_OPTION);
                                return;
                            }
                        }
                    
                        //store property values
                        try
                        {
                            Properties property = new Properties();
                            property.setProperty("m_hostName", hostName);
                            property.setProperty("m_port", port);
                            property.setProperty("m_DBName", dbName);
                            property.setProperty("m_userId", user);
                            property.setProperty("m_rdt",rdt);
                            property.setProperty("m_rid", rid);                            
                            property.store(new FileOutputStream("Viewer.properties"),
                                           "Viewer");
                        }
                        catch (Exception ee)
                        {
                        }
                        JGeoRaster jgeor=new JGeoRaster(m_conn,rdt,new NUMBER(rid));
                        m_viewer.setJGeoRaster(jgeor);
                        m_jgeor=jgeor;
                    }
                    catch (Exception ee)
                    {
                          ee.printStackTrace();
                          JOptionPane.showConfirmDialog(this, "Can't make a connection to DB",
                                                        "Error!", JOptionPane.DEFAULT_OPTION);
                          return;
                     }            
                }
            }
        }
    }
    
    static public void main(String[] args)
    {
        Tools tools = new Tools("GeoRaster Sample Tools");
        tools.setLocation(100, 100);
        tools.setResizable(false);
        tools.setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);
        tools.pack();
        tools.setVisible(true);        
    }

    public class GeoDBDialog extends JDialog
    {

        protected JTextField m_hostName=new JTextField();
        protected JTextField m_DBName=new JTextField();
        protected JTextField m_port=new JTextField();
        protected JTextField m_userId=new JTextField();
        protected JTextField m_password=new JPasswordField();
        protected JTextField m_rdt=new JTextField();
        protected JTextField m_rid=new JTextField();        

        public boolean m_ret=false;

        public GeoDBDialog(JFrame frame) throws HeadlessException
        {
            super();
            init(frame);
        }

        private void init(JFrame frame)
        {
            this.setModal(true);
            this.setTitle("Retrieve Raster from DB Dialog");
            this.setSize(500,250);
            this.setResizable(false);

            JPanel tmpPanel=new JPanel(new BorderLayout());
            JPanel mainPanel=new JPanel(new GridBagLayout());
            mainPanel.setBorder(new javax.swing.border.EmptyBorder(6,6,0,6));
            tmpPanel.add(mainPanel,BorderLayout.NORTH);
            JLabel hostNameL=new JLabel("Host name:"); 
            mainPanel.add(hostNameL,
                         new GridBagConstraints(0,0,1,1,0.0D,0.0D,
                         GridBagConstraints.CENTER,
                         GridBagConstraints.HORIZONTAL,
                         new Insets(3,6,3,3),0,0));
            m_hostName.setText("localhost"); 
            mainPanel.add(m_hostName,
                          new GridBagConstraints(1,0,4,1,1.0D,0.0D,
                          GridBagConstraints.CENTER,
                          GridBagConstraints.HORIZONTAL,
                          new Insets(3,3,3,6),0,0));
            hostNameL.setDisplayedMnemonic('h');
            hostNameL.setLabelFor(m_hostName);

            JLabel portL=new JLabel("Port number:");
            mainPanel.add(portL,
                          new GridBagConstraints(0,1,1,1,0.0D,0.0D,GridBagConstraints.CENTER,
                          GridBagConstraints.HORIZONTAL,
                          new Insets(3,6,3,3),0,0));
            m_port.setText("1521");
            mainPanel.add(m_port,
                          new GridBagConstraints(1,1,4,1,1.0D,0.0D,GridBagConstraints.CENTER,
                          GridBagConstraints.HORIZONTAL,
                          new Insets(3,3,3,6),0,0));
            portL.setDisplayedMnemonic('P');
            portL.setLabelFor(m_port);

            JLabel dbNameL=new JLabel("Database name:"); 
            mainPanel.add(dbNameL,
                          new GridBagConstraints(0,2,1,1,0.0D,0.0D,GridBagConstraints.CENTER,
                          GridBagConstraints.HORIZONTAL,
                          new Insets(3,6,3,3),0,0));
            m_DBName.setText("ORCL"); 
            mainPanel.add(m_DBName,
                          new GridBagConstraints(1,2,4,1,1.0D,0.0D,
                          GridBagConstraints.CENTER,
                          GridBagConstraints.HORIZONTAL,
                          new Insets(3,3,3,6),0,0));
            dbNameL.setDisplayedMnemonic('D');
            dbNameL.setLabelFor(m_DBName);

            JLabel userIdL=new JLabel("User name: ");
            mainPanel.add(userIdL,
                          new GridBagConstraints(0,3,1,1,0.0D,0.0D,GridBagConstraints.CENTER,
                          GridBagConstraints.HORIZONTAL,
                          new Insets(3,6,3,3),0,
                          0)); 
            mainPanel.add(m_userId,
                          new GridBagConstraints(1,3,4,1,1.0D,0.0D,
                          GridBagConstraints.CENTER,
                          GridBagConstraints.HORIZONTAL,
                          new Insets(3,3,3,6),0,0));
            userIdL.setDisplayedMnemonic('U');
            userIdL.setLabelFor(m_userId);

            JLabel passwordL=new JLabel("Password: "); 
            mainPanel.add(passwordL,
                          new GridBagConstraints(0,4,1,1,0.0D,0.0D,
                          GridBagConstraints.CENTER,
                          GridBagConstraints.HORIZONTAL,
                          new Insets(3,6,3,3),0,
                          0));
            mainPanel.add(m_password,
                          new GridBagConstraints(1,4,4,1,1.0D,0.0D,
                          GridBagConstraints.CENTER,
                          GridBagConstraints.HORIZONTAL,
                          new Insets(3,3,3,6),0,0));
            passwordL.setDisplayedMnemonic('A');
            passwordL.setLabelFor(m_password);
            
            JLabel rdtL=new JLabel("RDT: "); 
            mainPanel.add(rdtL,
                          new GridBagConstraints(0,5,1,1,0.0D,0.0D,
                          GridBagConstraints.CENTER,
                          GridBagConstraints.HORIZONTAL,
                          new Insets(3,6,3,3),0,
                          0));
            mainPanel.add(m_rdt,
                          new GridBagConstraints(1,5,4,1,1.0D,0.0D,
                          GridBagConstraints.CENTER,
                          GridBagConstraints.HORIZONTAL,
                          new Insets(3,3,3,6),0,0));
            passwordL.setLabelFor(m_rdt);            

            JLabel ridL=new JLabel("RID: "); 
            mainPanel.add(ridL,
                          new GridBagConstraints(0,6,1,1,0.0D,0.0D,
                          GridBagConstraints.CENTER,
                          GridBagConstraints.HORIZONTAL,
                          new Insets(3,6,3,3),0,
                          0));
            mainPanel.add(m_rid,
                          new GridBagConstraints(1,6,4,1,1.0D,0.0D,
                          GridBagConstraints.CENTER,
                          GridBagConstraints.HORIZONTAL,
                          new Insets(3,3,3,6),0,0));
            passwordL.setLabelFor(m_rid);     
            
            getContentPane().add(tmpPanel);

            JPanel butPanel=new JPanel(new GridBagLayout());
            JButton okButton=new JButton("  OK   ");
            JButton cancelButton=new JButton("Cancel");

            okButton.addActionListener(new java.awt.event.ActionListener()
                    {
                        public void actionPerformed(ActionEvent e)
                        {
                            okButton_actionPerformed(e);
                        }
                    });

            cancelButton.addActionListener(new java.awt.event.ActionListener()
                    {
                        public void actionPerformed(ActionEvent e)
                        {
                            cancelButton_actionPerformed(e);
                        }
                    });

            butPanel.add(okButton,
                         new GridBagConstraints(0,0,1,1,0.0D,0.0D,GridBagConstraints.CENTER,
                         GridBagConstraints.HORIZONTAL,
                         new Insets(0,0,6,3),0,
                         0));
            butPanel.add(cancelButton,
                         new GridBagConstraints(1,0,1,1,0.0D,0.0D,
                         GridBagConstraints.CENTER,
                         GridBagConstraints.HORIZONTAL,
                         new Insets(0,3,6,6),0,0));

            okButton.requestFocus();
            JPanel panel=new JPanel(new BorderLayout());
            panel.add(butPanel,BorderLayout.EAST);
            tmpPanel.add(panel,BorderLayout.SOUTH);

            okButton.setDefaultCapable(true);
            this.rootPane.setDefaultButton(okButton);

            //initialize text field values
            try
            {
                Properties property=new Properties();
                property.load(new FileInputStream("Viewer.properties"));
                if(property.getProperty("m_hostName")!=null)
                {
                    m_hostName.setText(property.getProperty("m_hostName"));
                }
                if(property.getProperty("m_DBName")!=null)
                {
                    m_DBName.setText(property.getProperty("m_DBName"));
                }
                if(property.getProperty("m_port")!=null)
                {
                    m_port.setText(property.getProperty("m_port"));
                }
                if(property.getProperty("m_userId")!=null)
                {
                    m_userId.setText(property.getProperty("m_userId"));
                }
                if(property.getProperty("m_rdt")!=null)
                {
                    m_rdt.setText(property.getProperty("m_rdt"));
                }
                if(property.getProperty("m_rid")!=null)
                {
                    m_rid.setText(property.getProperty("m_rid"));
                }                
            }
            catch(Exception ee)
            {
            }
            this.setLocation(frame.getLocation().x+150,
                             frame.getLocation().y+150);
        }

        void okButton_actionPerformed(ActionEvent e)
        {
            // check the validity of the entered values.
            String msg=checkForNullValues();
            if(msg!=null)
            {
                JOptionPane.showConfirmDialog(this,"Missing values for : "+msg,
                                              "Error!",
                                              javax.swing.JOptionPane.DEFAULT_OPTION);
                return;
            }
            m_ret=true;
            dispose();
        }

        void cancelButton_actionPerformed(ActionEvent e)
        {
            dispose();
        }

        // check any not-null constraint violations.
        // returns a string of comma delimited fields that are null

        private String checkForNullValues()
        {
            String data=new String();
            StringBuffer msg=new StringBuffer();
            int count=0;

            data=m_hostName.getText();
            if(data==null||data.trim().length()==0)
            {
                if(count!=0)
                {
                    msg.append(", "+"Host name");
                }
                else
                {
                    msg.append("Host name");
                }
                count++;
            }

            data=m_port.getText();
            if(data==null||data.trim().length()==0)
            {
                if(count!=0)
                {
                    msg.append(", "+"Port Number");
                }
                else
                {
                    msg.append("Port Number");
                }
                count++;
            }

            data=m_DBName.getText();
            if(data==null||data.trim().length()==0)
            {
                if(count!=0)
                {
                    msg.append(", "+"Database name");
                }
                else
                {
                    msg.append("Database name");
                }
                count++;
            }

            data=m_userId.getText();
            if(data==null||data.trim().length()==0)
            {
                if(count!=0)
                {
                    msg.append(", "+"User ID");
                }
                else
                {
                    msg.append("User ID");
                }
                count++;
            }

            data=m_password.getText();
            if(data==null||data.trim().length()==0)
            {
                if(count!=0)
                {
                    msg.append(", "+"Password");
                }
                else
                {
                    msg.append("Password");
                }
                count++;
            }

            if(count==0)
            {
                return null;
            }
            else
            {
                return msg.toString();
            }
        }
    }
}
